/** @file   map.cpp
 * @brief   Implementation of Map class.
 * @version $Revision: 1.2 $
 * @date    $Date: 2006/08/13 10:50:23 $
 * @author  Tomi Lamminsaari
 */

#include "www_map.h"
#include "mappyal.h"
#include "vec2d.h"
#include "filehasher.h"
#include "utils.h"
#include "warglobals.h"
#include "consts.h"
#include "www_assert.h"
using std::ifstream;
using std::istream;
using std::string;
using std::vector;
using eng2d::Vec2D;
using eng2d::FileHasher;

namespace WeWantWar {


Map::Block Map::datablock;
int Map::scrollX = 0;
int Map::scrollY = 0;
int Map::sectorWidth = 10;
int Map::sectorHeight = 10;
int Map::mwidth = 0;
int Map::mheight = 0;



/** Converts the integer to layerid - value
 */
Map::TLayerId Map::int2LayerId( int aLayerNum )
{
  switch ( aLayerNum ) {
    case ( 0 ): {
      return EBackground;
    }
    case ( 1 ): {
      return EObjects;
    }
    case ( 2 ): {
      return EDecorative;
    }
    case ( 3 ): {
      return EStorage;
    }
    case ( 4 ): {
      return EDynamicShadows;
    }
    default: {
      return EBackground;
    }
  }
}



/** Loads the map
 */
int Map::load( std::string file )
{
  if ( MapLoad( file.c_str() ) != 0 ) {
    return -1;
  }
  
  sectorWidth = getWidth( IN_PIXELS ) / Consts::SECTOR_COLUMNS;
  sectorHeight = getHeight( IN_PIXELS ) / Consts::SECTOR_ROWS;
  mwidth = getWidth( IN_PIXELS );
  mheight = getHeight( IN_PIXELS );
  Map::init();
  return 0;
}



/** Releases the resources allocated for the map
 */
void Map::destroy()
{
  MapFreeMem();
}



/** Selects the layer
 * Param      layer               : the layer
 * Returns
 * Exception
 */
int Map::selectLayer( TLayerId layer )
{
  return MapChangeLayer(layer);
}




/** Returns the block at given coordinates.
 */
Map::Block* Map::blockAt(int x, int y, DimUnit unit)
{
  WWW_ASSERT( x >= 0 && y >= 0 );
  WWW_ASSERT( x < Map::getWidth( unit ) && y < Map::getHeight( unit ) );
  BLKSTR* pB = 0;
  if ( unit == IN_PIXELS ) {
    pB = MapGetBlockInPixels(x,y);
  } else {
    pB = MapGetBlock(x, y);
  }
  
  datablock.data = pB->bgoff;
  datablock.data2 = pB->fgoff;
  datablock.user1 = pB->user1;
  datablock.user2 = pB->user2;
  datablock.user3 = pB->user3;
  datablock.user4 = pB->user4;
  datablock.user5 = pB->user5;
  datablock.user6 = pB->user6;
  datablock.user7 = pB->user7;
  datablock.tl = pB->tl;
  datablock.tr = pB->tr;
  datablock.bl = pB->bl;
  datablock.br = pB->br;
  datablock.trigger = pB->trigger;
  return &datablock;
}




/** Sets the block to the given coordinates
 */
void Map::setBlock(int x, int y, DimUnit unit, int data)
{
  if ( unit == IN_PIXELS ) {
    MapSetBlockInPixels(x, y, data);
  } else {
    MapSetBlock(x, y, data);
  }
}



/** Returns the block number at given position
 */
int Map::blockNumberAt( int x, int y, DimUnit unit )
{
  if ( unit == IN_PIXELS ) {
    return MapGetBlockNumberInPixels( x, y );
  } else {
    return MapGetBlockNumber( x, y );
  }
}



/** Returns the width of the current map in blocks
 */
int Map::getWidth(DimUnit unit)
{
  if (unit == IN_PIXELS) {
    return mapwidth * getBlockWidth();
  }
  return mapwidth;
}



/** Returns the height of the current map in blocks
 */
int Map::getHeight(DimUnit unit)
{
  if (unit == IN_PIXELS) {
    return mapheight * getBlockHeight();
  }
  return mapheight;
}



/** Returns the width of the blocks
 */
int Map::getBlockWidth()
{
  return mapblockwidth;
}



/** Returns the height of the blocks
 */
int Map::getBlockHeight()
{
  return mapblockheight;
}



/** Redraws the map
 */
void Map::redraw(int fromX, int fromY, int toX, int toY, int w, int h, bool t)
{
  // Make sure that we don't try to draw outside the map
  if ( fromX < 0 ) fromX = 0;
  if ( fromY < 0 ) fromY = 0;

  if ( fromX + w > getWidth( IN_PIXELS ) ) {
    fromX = getWidth( IN_PIXELS ) - w;
  }
  if ( fromY + h > getHeight( IN_PIXELS ) ) {
    fromY = getHeight( IN_PIXELS ) - h;
  }

  if ( t == true ) {
    MapDrawBGT( eng2d::Display::buffer, fromX,fromY, toX,toY, w,h );
    
  } else {

    MapDrawBG( eng2d::Display::buffer, fromX, fromY, toX, toY, w, h );
  }
}



/** Redraws the foreground
 */
void Map::redrawFG(int fromX, int fromY, int toX, int toY, int w, int h)
{
  // Make sure that we don't try to draw outside the map
  if ( fromX < 0 ) fromX = 0;
  if ( fromY < 0 ) fromY = 0;

  if ( fromX + w > getWidth( IN_PIXELS ) ) {
    fromX = getWidth( IN_PIXELS ) - w;
  }
  if ( fromY + h > getHeight( IN_PIXELS ) ) {
    fromY = getHeight( IN_PIXELS ) - h;
  }

  MapDrawFG( eng2d::Display::buffer, fromX, fromY, toX, toY, w, h, 0 );
}



/** Updates the animations
 */
void Map::update()
{
  // update the map animations
  MapUpdateAnims();
}



/** Initializes the map. This will be called automatically after the loading
 * of a new map.
 */
void Map::init()
{
  MapInitAnims();
}



/** Tells is the block in given coordinates passable or not
 */
bool Map::collide(int x, int y, DimUnit unit)
{
  BLKSTR* pB = 0;
  if ( unit == IN_PIXELS ) {
    if ( x < 2 || y < 2 || x > getWidth( IN_PIXELS ) - 2 ||
         y > getHeight(IN_PIXELS) - 2 ) {
      return true;
    }
    pB = MapGetBlockInPixels(x,y);
  } else {
    pB = MapGetBlock(x, y);
  }
  
  if ( pB->tl ) {
    return true;
  }
  return false;
}



/** As above, but given vector has the coordinates in pixels
 */
bool Map::collide( const eng2d::Vec2D& p )
{
  return Map::collide( static_cast<int>(p.x()), static_cast<int>(p.y()),
                       IN_PIXELS );
}



/** Checks the bullet collision on given coordinate.
 */
bool Map::bulletCollide( const eng2d::Vec2D& p )
{
  if ( p.x() < 2 || p.y() < 2 || p.x() > getWidth( IN_PIXELS ) - 2 ||
       p.y() > getHeight(IN_PIXELS) - 2 ) {
    return true;
  }
    
  BLKSTR* pB = 0;
  pB = MapGetBlockInPixels( static_cast<int>( p.x() ), static_cast<int>( p.y() ) );
  if ( pB->tr ) {
    return true;
  }
  return false;
}



/** Tells if there is free way from p1 to p2
 */
bool Map::lineBlocked(const eng2d::Vec2D& p1, const eng2d::Vec2D& p2)
{
  eng2d::Vec2D dirV( p1 );
  dirV -= p2;
  int iterRounds = static_cast<int>( dirV.length() );
  dirV.norm();

  eng2d::Vec2D tmpPos( p1 );
  for ( int i=0; i < iterRounds; i++ ) {
    if ( collide( tmpPos ) == true ) {
      return true;
    }
  }
  return false;
}



/** Tells if there's a free way from p1 to p2
 */
bool Map::lineBlocked( const eng2d::Vec2D& rP1, const eng2d::Vec2D& rP2,
                       float stepSize, float maxDistance )
{
}


/** Default constructor
 */
Map::Map()
{
}



/** Destructor
 */
Map::~Map()
{
}

} // end of namespace
